import xbmc
from urllib.request import Request, urlopen
from urllib.parse import quote
from urllib.error import URLError
import time
import json

# Volume map: Maps /Volumes/X/ to the server's expected path
VOLUME_MAP = {
    "3": "/mnt/sda1/Movies3/",
    "4": "/mnt/sda1/Movies4/",
    "5": "/mnt/sda1/Movies5/",
    "7": "/mnt/sda1/Movies7/",
    "8": "/mnt/sda1/Movies8/"
}

# Sony Bravia TV configuration
BRAVIA_IP = "192.168.1.55"
BRAVIA_PSK = "0000"  # Confirmed working

# Oppo device HTTP configuration
OPPO_IP = "192.168.1.57"
OPPO_PORT = 436

class OppoPlayer(xbmc.Player):
    def __init__(self):
        super().__init__()
        self.is_playing_external = False
        self.monitor = xbmc.Monitor()
        xbmc.log("OppoPlayer initialized", xbmc.LOGINFO)

    def switch_to_hdmi4(self):
        url = f"http://{BRAVIA_IP}/sony/avContent"
        payload = '{"method":"setPlayContent","params":[{"uri":"extInput:hdmi?port=4"}],"id":1,"version":"1.0"}'
        headers = {"Content-Type": "application/json", "X-Auth-PSK": BRAVIA_PSK}
        
        try:
            req = Request(url, data=payload.encode('utf-8'), headers=headers, method="POST")
            xbmc.log(f"Sending HDMI switch request to Bravia TV (HDMI 4): {url}", xbmc.LOGINFO)
            with urlopen(req, timeout=5) as response:
                status_code = response.getcode()
                response_content = response.read().decode('utf-8', errors='ignore')
                xbmc.log(f"Bravia HDMI switch response: status code {status_code}, content: {response_content}", xbmc.LOGINFO)
                if status_code != 200:
                    xbmc.log(f"HDMI switch to HDMI 4 failed with status {status_code}", xbmc.LOGERROR)
                    return False
                time.sleep(1)  # Stabilize HDMI switch
                return True
        except URLError as e:
            xbmc.log(f"Failed to switch HDMI input to HDMI 4: {e}", xbmc.LOGERROR)
            return False

    def switch_to_hdmi2(self):
        url = f"http://{BRAVIA_IP}/sony/avContent"
        payload = '{"method":"setPlayContent","params":[{"uri":"extInput:hdmi?port=2"}],"id":1,"version":"1.0"}'
        headers = {"Content-Type": "application/json", "X-Auth-PSK": BRAVIA_PSK}
        
        try:
            req = Request(url, data=payload.encode('utf-8'), headers=headers, method="POST")
            xbmc.log(f"Sending HDMI switch request to Bravia TV (HDMI 2): {url}", xbmc.LOGINFO)
            with urlopen(req, timeout=5) as response:
                status_code = response.getcode()
                response_content = response.read().decode('utf-8', errors='ignore')
                xbmc.log(f"Bravia HDMI switch response: status code {status_code}, content: {response_content}", xbmc.LOGINFO)
                if status_code != 200:
                    xbmc.log(f"HDMI switch to HDMI 2 failed with status {status_code}", xbmc.LOGERROR)
                    return False
                time.sleep(1)  # Stabilize HDMI switch
                return True
        except URLError as e:
            xbmc.log(f"Failed to switch HDMI input to HDMI 2: {e}", xbmc.LOGERROR)
            return False

    def check_oppo_playback_status(self):
        url = f"http://{OPPO_IP}:{OPPO_PORT}/getmovieplayinfo"
        try:
            req = Request(url)
            with urlopen(req, timeout=5) as response:
                status_code = response.getcode()
                response_content = response.read().decode('utf-8', errors='ignore')
                xbmc.log(f"Oppo HTTP status response: status code: {status_code}, content: {response_content}", xbmc.LOGINFO)
                return response_content
        except URLError as e:
            xbmc.log(f"HTTP status request failed: {e}", xbmc.LOGERROR)
            return None

    def monitor_oppo_playback(self):
        xbmc.log("Starting Oppo playback monitoring", xbmc.LOGINFO)
        
        # Initial check to confirm Oppo starts playing
        start_time = time.time()
        timeout = 15  # Increased timeout to 15 seconds
        max_retries = 3  # Retry up to 3 times for failed requests
        retry_count = 0
        while time.time() - start_time < timeout and not self.monitor.abortRequested():
            response = self.check_oppo_playback_status()
            if response:
                try:
                    status_data = json.loads(response)
                    xbmc.log(f"Oppo status data: {status_data}", xbmc.LOGDEBUG)
                    success = status_data.get("success", True)
                    play_status = status_data.get("playinfo", {}).get("e_play_status", -1)
                    if success and play_status == 0:
                        xbmc.log("Oppo confirmed playing, entering monitoring loop", xbmc.LOGINFO)
                        break
                    else:
                        xbmc.log(f"Oppo not yet playing: success={success}, e_play_status={play_status}", xbmc.LOGINFO)
                        retry_count += 1
                        if retry_count >= max_retries:
                            xbmc.log(f"Oppo failed to start after {max_retries} retries", xbmc.LOGERROR)
                            self.is_playing_external = False
                            self.stop_and_cleanup()
                            return
                except json.JSONDecodeError as e:
                    xbmc.log(f"Failed to parse Oppo status response as JSON: {e}, response: {response}", xbmc.LOGWARNING)
                    retry_count += 1
                    if retry_count >= max_retries:
                        xbmc.log(f"JSON decode error after {max_retries} retries", xbmc.LOGERROR)
                        self.is_playing_external = False
                        self.stop_and_cleanup()
                        return
            else:
                xbmc.log("No response from Oppo, retrying", xbmc.LOGWARNING)
                retry_count += 1
                if retry_count >= max_retries:
                    xbmc.log(f"No response from Oppo after {max_retries} retries", xbmc.LOGERROR)
                    self.is_playing_external = False
                    self.stop_and_cleanup()
                    return
            time.sleep(1)
        else:
            xbmc.log("Oppo failed to start playback within timeout or Kodi aborted", xbmc.LOGERROR)
            self.is_playing_external = False
            self.stop_and_cleanup()
            return

        # Main monitoring loop
        retry_count = 0
        while not self.monitor.abortRequested():
            response = self.check_oppo_playback_status()
            if response:
                try:
                    status_data = json.loads(response)
                    xbmc.log(f"Oppo status data: {status_data}", xbmc.LOGDEBUG)
                    success = status_data.get("success", True)
                    if success:
                        play_status = status_data.get("playinfo", {}).get("e_play_status", -1)
                        if play_status == 0:
                            xbmc.log("Oppo is playing", xbmc.LOGINFO)
                            retry_count = 0
                        elif play_status == 56:
                            xbmc.log("Oppo is paused", xbmc.LOGINFO)
                            retry_count = 0
                        else:
                            xbmc.log(f"Oppo in unexpected play state: e_play_status={play_status}", xbmc.LOGWARNING)
                            retry_count += 1
                            if retry_count >= max_retries:
                                xbmc.log(f"Oppo in unexpected state for {max_retries} checks, stopping", xbmc.LOGERROR)
                                self.is_playing_external = False
                                self.stop_and_cleanup()
                                break
                    elif not success and status_data.get("errcode1", 0) == -5:
                        xbmc.log("Detected playback stopped on Oppo (errcode1=-5)", xbmc.LOGINFO)
                        self.is_playing_external = False
                        self.stop_and_cleanup()
                        break
                    else:
                        xbmc.log(f"Oppo returned unsuccessful response: {status_data}", xbmc.LOGWARNING)
                        retry_count += 1
                        if retry_count >= max_retries:
                            xbmc.log(f"Oppo returned unsuccessful response for {max_retries} checks, stopping", xbmc.LOGERROR)
                            self.is_playing_external = False
                            self.stop_and_cleanup()
                            break
                except json.JSONDecodeError as e:
                    xbmc.log(f"Failed to parse Oppo status response as JSON: {e}, response: {response}", xbmc.LOGWARNING)
                    retry_count += 1
                    if retry_count >= max_retries:
                        xbmc.log(f"JSON decode error for {max_retries} checks, stopping", xbmc.LOGERROR)
                        self.is_playing_external = False
                        self.stop_and_cleanup()
                        break
            else:
                xbmc.log("No response from Oppo, retrying", xbmc.LOGWARNING)
                retry_count += 1
                if retry_count >= max_retries:
                    xbmc.log(f"No response from Oppo for {max_retries} checks, stopping", xbmc.LOGERROR)
                    self.is_playing_external = False
                    self.stop_and_cleanup()
                    break
            time.sleep(7)
        xbmc.log("Oppo playback monitoring stopped", xbmc.LOGINFO)

    def onPlayBackStarted(self):
        xbmc.log("Playback started detected", xbmc.LOGINFO)
        
        xbmc.sleep(500)
        
        try:
            file_path = self.getPlayingFile()
            xbmc.log(f"Original file path: {file_path}", xbmc.LOGINFO)
            
            normalized_path = file_path
            if file_path.startswith("nfs://"):
                path_parts = file_path.split("/", 4)
                if len(path_parts) > 4 and path_parts[3] == "Volumes":
                    volume_num = path_parts[4].split("/")[0]
                    if volume_num in VOLUME_MAP:
                        normalized_path = VOLUME_MAP[volume_num] + "/".join(path_parts[4].split("/")[1:])
                        xbmc.log(f"Normalized NFS file path: {normalized_path}", xbmc.LOGINFO)
                    else:
                        xbmc.log(f"Volume {volume_num} not in VOLUME_MAP, aborting", xbmc.LOGERROR)
                        self.stop_and_cleanup()
                        return
                else:
                    xbmc.log("NFS path does not contain /Volumes/, aborting", xbmc.LOGERROR)
                    self.stop_and_cleanup()
                    return
            else:
                xbmc.log("File path not under nfs://, aborting", xbmc.LOGERROR)
                self.stop_and_cleanup()
                return

            if not self.switch_to_hdmi4():
                xbmc.log("HDMI switch to Oppo failed, aborting", xbmc.LOGERROR)
                self.stop_and_cleanup()
                return

            json_payload = f'{{"path":"{normalized_path}","index":0,"type":1,"appDeviceType":2,"extraNetPath":"192.168.1.238/Storage/UHD/","playMode":0}}'
            encoded_payload = quote(json_payload)
            url = f"http://{OPPO_IP}:{OPPO_PORT}/playnormalfile?{encoded_payload}"
            xbmc.log(f"Constructed URL: {url}", xbmc.LOGINFO)

            time.sleep(1)  # Ensure Oppo is ready
            for attempt in range(2):
                try:
                    req = Request(url)
                    xbmc.log(f"Sending HTTP request (attempt {attempt + 1})", xbmc.LOGINFO)
                    with urlopen(req, timeout=5) as response:
                        status_code = response.getcode()
                        response_content = response.read().decode('utf-8', errors='ignore')
                        xbmc.log(f"HTTP response: status code: {status_code}, content: {response_content}", xbmc.LOGINFO)
                        if status_code == 200 and '"success":true' in response_content:
                            xbmc.log("Server confirmed successful playback", xbmc.LOGINFO)
                            self.is_playing_external = True
                            # Stop Kodi playback but don't switch HDMI yet
                            xbmc.executebuiltin("PlayerControl(Stop)")
                            xbmc.executebuiltin("Dialog.Close(busydialog)")
                            xbmc.executebuiltin("ActivateWindow(home)")
                            self.monitor_oppo_playback()
                            return
                        else:
                            xbmc.log("Server rejected request (success:false)", xbmc.LOGWARNING)
                except URLError as e:
                    xbmc.log(f"HTTP request failed: {e}", xbmc.LOGERROR)
                time.sleep(1)
            xbmc.log("Failed to start playback on Oppo after retries", xbmc.LOGERROR)
            self.stop_and_cleanup()
        except Exception as e:
            xbmc.log(f"Error in onPlayBackStarted: {e}", xbmc.LOGERROR)
            self.stop_and_cleanup()

    def stop_and_cleanup(self):
        xbmc.log(f"stop_and_cleanup called, is_playing_external={self.is_playing_external}", xbmc.LOGINFO)
        xbmc.executebuiltin("PlayerControl(Stop)")
        time.sleep(0.5)
        xbmc.executebuiltin("Dialog.Close(busydialog)")
        xbmc.executebuiltin("ActivateWindow(home)")
        if not self.is_playing_external:
            xbmc.log("Oppo not playing, switching to HDMI 2", xbmc.LOGINFO)
            self.switch_to_hdmi2()
        else:
            xbmc.log("Oppo is playing, keeping HDMI 4", xbmc.LOGINFO)
        xbmc.log("Completed stop_and_cleanup", xbmc.LOGINFO)

    def onPlayBackStopped(self):
        xbmc.log(f"onPlayBackStopped called, is_playing_external={self.is_playing_external}", xbmc.LOGINFO)
        if not self.is_playing_external:
            xbmc.log("Kodi playback stopped, ensuring cleanup", xbmc.LOGINFO)
            self.stop_and_cleanup()
        else:
            xbmc.log("Kodi playback stopped, but Oppo is playing", xbmc.LOGINFO)

if __name__ == "__main__":
    player = OppoPlayer()
    while not player.monitor.abortRequested():
        player.monitor.waitForAbort(1)